const express = require("express");
const fs = require("fs");
const path = require("path");
const router = express.Router();
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const crypto = require("crypto");
const asyncHandler = require("express-async-handler");
const { JWT_SECRET } = require("../env.json");
const User = require("../model/user");
const authValidator = require("../middleware/authValidator");
const { uploadImage } = require("../helper/upload");
const {
  generateForgotPasswordEmailBody,
} = require("../helper/generateEmailContent");
const { sendMail } = require("../helper/sendMail");
const mongoose = require("mongoose");
const { UNSAFE_useScrollRestoration } = require("react-router-dom");

// ======================================
// REGISTER
// ======================================
router.post(
  "/register",
  uploadImage.single("pfp"),
  asyncHandler(async (req, res) => {
    const {
      firstName,
      lastName,
      username,
      email,
      password,
      shortBio,
      occupation,
      favouriteGenres,
      role,
    } = req.body;

    if (!username || !email || !password || !role) {
      res.status(400);
      throw new Error("Fill all required fields");
    }

    // Check if email already exists
    const emailExists = await User.findOne({ email });
    if (emailExists) {
      res.status(400);
      throw new Error("User with this email already exists");
    }

    // Check if username already exists
    const usernameExists = await User.findOne({ username });
    if (usernameExists) {
      res.status(400);
      throw new Error("User with this username already exists");
    }

    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashPassword = await bcrypt.hash(password, salt);

    // Handle profile picture
    let profilePicPath = null;
    if (req.file) {
      profilePicPath = `/uploads/images/${req.file.filename}`;
    }

    // Fix: normalize favouriteGenres
    let parsedGenres = [];
    if (favouriteGenres) {
      try {
        parsedGenres =
          typeof favouriteGenres === "string"
            ? JSON.parse(favouriteGenres)
            : favouriteGenres;

        parsedGenres = parsedGenres.map(
          (id) => new mongoose.Types.ObjectId(id)
        );
      } catch (err) {
        console.error("Invalid favouriteGenres format:", favouriteGenres);
        res.status(400);
        throw new Error("Invalid favouriteGenres data");
      }
    }

    // Create user
    await User.create({
      firstName,
      lastName,
      username,
      email,
      password: hashPassword,
      shortBio,
      occupation,
      favouriteGenres: parsedGenres,
      role,
      pfp: profilePicPath,
    });

    res.status(201).json({
      success: true,
      message: "User Registered Successfully. Please log in.",
    });
  })
);

// ======================================
// LOGIN
// ======================================
router.post(
  "/login",
  asyncHandler(async (req, res) => {
    const { emailOrUsername, password, from } = req.body;
    const isWeb = from === "web";

    if (!emailOrUsername || !password) {
      res.status(400);
      throw new Error("Fill all fields");
    }

    const user = await User.findOne({
      $or: [{ email: emailOrUsername }, { username: emailOrUsername }],
    });

    if (!user) {
      res.status(400);
      throw new Error("Invalid Credentials");
    }

    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) {
      res.status(400);
      throw new Error("Invalid Credentials");
    }

    const userPayload = {
      id: user._id,
      email: user.email,
      username: user.username,
      firstName: user.firstName,
      lastName: user.lastName,
      role: user.role,
    };

    const authtoken = jwt.sign(userPayload, JWT_SECRET, { expiresIn: "1d" });

    if (isWeb) {
      return res.status(200).json({
        success: true,
        message: "User Logged In Successfully",
        authtoken,
      });
    }

    // Cookie options for admin login
    const cookieOptions = {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production", // true in production
      sameSite: "lax",
      maxAge: 24 * 60 * 60 * 1000, // 1 day
    };

    return res
      .status(200)
      .cookie("authtoken", authtoken, cookieOptions)
      .redirect(`/admin?message=User Logged In Successfully`);
  })
);

// ======================================
// LOGOUT
// ======================================
router.post(
  "/logout",
  authValidator(),
  asyncHandler(async (req, res) => {
    const cookieAuthtoken = req.cookies?.authtoken;

    if (!cookieAuthtoken) {
      return res.status(400).json({
        success: false,
        message: "No active session found ",
      });
    }

    res.clearCookie("authtoken", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production", // true in prod (HTTPS)
      sameSite: "lax",
      path: "/", // must match your login cookie path
    });

    // Optional: For clients expecting immediate state update
    res.status(200).json({
      success: true,
      message: "Logged out successfully",
    });
  })
);

// ==========================
// EDIT PROFILE (Authenticated)
// ==========================
router.put(
  "/edit-profile",
  authValidator(),
  uploadImage.single("pfp"),
  asyncHandler(async (req, res) => {
    const {
      firstName,
      lastName,
      username,
      email,
      shortBio,
      occupation,
      favouriteGenres,
    } = req.body;
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: "User not found" });

    // Handle profile picture update
    if (req.file) {
      if (user.pfp && user.pfp !== "/uploads/images/default.jpg") {
        const oldPath = path.join(__dirname, "..", user.pfp);
        if (fs.existsSync(oldPath)) fs.unlinkSync(oldPath);
      }
      user.pfp = `/uploads/images/${req.file.filename}`;
    }

    // Handle favouriteGenres parsing
    let genres = user.favouriteGenres;
    if (favouriteGenres) {
      try {
        const parsed =
          typeof favouriteGenres === "string"
            ? JSON.parse(favouriteGenres)
            : favouriteGenres;
        genres = parsed
          .filter((id) => mongoose.Types.ObjectId.isValid(id))
          .map((id) => new mongoose.Types.ObjectId(id));
      } catch {
        return res
          .status(400)
          .json({ message: "Invalid favouriteGenres format" });
      }
    }

    // Update fields
    Object.assign(user, {
      firstName,
      lastName,
      username,
      email,
      shortBio,
      occupation,
      favouriteGenres: genres,
    });

    await user.save();

    // Generate new token
    const payload = {
      id: user._id,
      username: user.username,
      email: user.email,
      role: user.role,
    };
    const token = jwt.sign(payload, JWT_SECRET, { expiresIn: "1h" });

    res.cookie("authtoken", token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      path: "/",
    });

    res.status(200).json({
      success: true,
      message: "Profile updated successfully",
      user,
    });
  })
);

// ======================================
// FORGOT PASSWORD
// ======================================
router.post(
  "/forgot-password",
  asyncHandler(async (req, res) => {
    const { email } = req.body;

    if (!email) {
      res.status(400);
      throw new Error("Email is required");
    }

    // Find user by email
    const user = await User.findOne({ email });

    if (!user) {
      // Don't reveal if user exists or not for security
      return res.status(200).json({
        success: true,
        message:
          "If an account exists with this email, you will receive a password reset link.",
      });
    }

    // Generate reset token
    const resetToken = crypto.randomBytes(32).toString("hex");
    const hashedToken = crypto
      .createHash("sha256")
      .update(resetToken)
      .digest("hex");

    // Save hashed token and expiry to user document
    user.resetPasswordToken = hashedToken;
    user.resetPasswordExpires = Date.now() + 3600000; // 1 hour
    await user.save();

    // Create reset URL
    const resetUrl = `${req.protocol}://${req.get(
      "host"
    )}/reset-password/${resetToken}`;

    try {
      // Generate email content
      const { subject, html } = generateForgotPasswordEmailBody({
        email: user.email,
        resetToken,
        resetUrl,
        expiresIn: "1 hour",
      });

      // Send email
      await sendMail(user.email, subject, html);

      res.status(200).json({
        success: true,
        message: "Password reset link has been sent to your email.",
      });
    } catch (error) {
      // Clear reset token if email fails
      user.resetPasswordToken = undefined;
      user.resetPasswordExpires = undefined;
      await user.save();

      console.error("Email sending failed:", error);
      res.status(500);
      throw new Error(
        "Failed to send password reset email. Please try again later."
      );
    }
  })
);

// ======================================
// RESET PASSWORD
// ======================================
router.post(
  "/reset-password/:token",
  asyncHandler(async (req, res) => {
    const { token } = req.params;
    const { password } = req.body;

    if (!password) {
      res.status(400);
      throw new Error("Password is required");
    }

    if (password.length < 6) {
      res.status(400);
      throw new Error("Password must be at least 6 characters long");
    }

    // Hash the token from URL to compare with stored hash
    const hashedToken = crypto.createHash("sha256").update(token).digest("hex");

    // Find user with valid token and not expired
    const user = await User.findOne({
      resetPasswordToken: hashedToken,
      resetPasswordExpires: { $gt: Date.now() },
    });

    if (!user) {
      res.status(400);
      throw new Error("Invalid or expired reset token");
    }

    // Hash new password
    const salt = await bcrypt.genSalt(10);
    const hashPassword = await bcrypt.hash(password, salt);

    // Update password and clear reset fields
    user.password = hashPassword;
    user.resetPasswordToken = undefined;
    user.resetPasswordExpires = undefined;
    await user.save();

    res.status(200).json({
      success: true,
      message:
        "Password reset successful. You can now login with your new password.",
    });
  })
);

module.exports = router;
